#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/time.h>
#include <termios.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include "ACTS2200.h"

// Linux-friendly kbhit()
int kbhit(void) {
    struct termios oldt, newt;
    int ch;
    int oldf;
    tcgetattr(STDIN_FILENO, &oldt);
    newt = oldt;
    newt.c_lflag &= ~(ICANON | ECHO);
    tcsetattr(STDIN_FILENO, TCSANOW, &newt);
    oldf = fcntl(STDIN_FILENO, F_GETFL, 0);
    fcntl(STDIN_FILENO, F_SETFL, oldf | O_NONBLOCK);
    ch = getchar();
    tcsetattr(STDIN_FILENO, TCSANOW, &oldt);
    fcntl(STDIN_FILENO, F_SETFL, oldf);
    if (ch != EOF) {
        ungetc(ch, stdin);
        return 1;
    }
    return 0;
}

// now time (s)
double now_sec() {
    struct timeval tv;
    gettimeofday(&tv, NULL);
    return tv.tv_sec + tv.tv_usec / 1e6;
}

int main(void) {
    // ---------- USER SETTINGS ----------
    const double sampRate    = 10000.0;     // Hz
    const unsigned chunk     = 100;         // points per read
    const unsigned aiChan    = 0;           // AI0
    const double   sens_mV_g = 500.0;       // accelerometer sensitivity (mV/g)
    // -----------------------------------

    // ----- 文件逻辑已被移除 -----

    HANDLE hDev;
    ACTS2200_AI_PARAM ai;
    double *buf = NULL;
    unsigned long long total = 0;
    double t0, t1;

    // 状态消息打印到 stderr，这样它们就不会和数据混在一起
    fprintf(stderr, "Opening device...\n");
    hDev = ACTS2200_DEV_Create(0);
    if (hDev == INVALID_HANDLE_VALUE) { fprintf(stderr, "Open device failed\n"); return -1; }

    // Configure AI
    memset(&ai, 0, sizeof(ai));
    ai.nSampChanCount = 1;
    ai.CHParam[0].nChannel   = aiChan;
    ai.CHParam[0].nSampRange = ACTS2200_AI_SAMPRANGE_N10_P10V; // +/- 10V
    ai.CHParam[0].nRefGround = ACTS2200_AI_REFGND_RSE;         // single-ended
    ai.nSampMode     = ACTS2200_SAMPMODE_CONTINUOUS;
    ai.nSampsPerChan = 4096;
    ai.fSampRate     = sampRate;
    ai.nSampClkSrc   = ACTS2200_VAL_LOCAL;
    ai.StartTrig.nTrigType = ACTS2200_VAL_NONE;

    if (!ACTS2200_AI_InitTask(hDev, &ai)) { fprintf(stderr, "AI_InitTask failed\n"); ACTS2200_DEV_Release(hDev); return -2; }
    if (!ACTS2200_AI_StartTask(hDev))     { fprintf(stderr, "AI_StartTask failed\n"); ACTS2200_AI_ReleaseTask(hDev); ACTS2200_DEV_Release(hDev); return -3; }

    // compute scale
    const double sens_V_g = sens_mV_g / 1000.0; // V per g
    const double g_to_mps2 = 9.80665;

    buf = (double*)malloc(sizeof(double) * chunk);
    if (!buf) { fprintf(stderr, "No memory\n"); ACTS2200_AI_StopTask(hDev); ACTS2200_AI_ReleaseTask(hDev); ACTS2200_DEV_Release(hDev); return -5; }

    fprintf(stderr, "Start continuous acquisition...\n");
    fprintf(stderr, "Sampling rate = %.1f Hz, chunk = %u points\n", sampRate, chunk);
    fprintf(stderr, "Press any key to stop.\n");

    // 打印表头到 stdout
    printf("Time(s),Volt(V),Accel(g),Accel(m/s^2)\n");
    fflush(stdout);

    t0 = now_sec();
    while (!kbhit()) {
        unsigned got = 0;
        if (!ACTS2200_AI_ReadAnalog(hDev, buf, chunk, &got, 2.0, ACTS2200_FILLMODE_GroupByScanNumber)) {
            fprintf(stderr, "Read timeout/error\n");
            continue;
        }
        t1 = now_sec();
        double elapsed = t1 - t0;
        double dt = 1.0 / sampRate;

        for (unsigned i = 0; i < got; ++i) {
            double t = elapsed + i * dt;
            double v = buf[i];
            double acc_g = v / sens_V_g;
            double acc_mps2 = acc_g * g_to_mps2;

            // 打印数据到 stdout
            printf("%.6f,%.6f,%.6f,%.6f\n", t, v, acc_g, acc_mps2);
        }

        // 刷新 stdout 来确保 Python 立刻收到
        fflush(stdout);
        total += got;
    }

    double T = now_sec() - t0;
    fprintf(stderr, "\nStopping...\n");
    fprintf(stderr, "Total points: %llu\n", total);
    fprintf(stderr, "Total time: %.3f seconds\n", T);
    fprintf(stderr, "Approx. effective rate: %.1f Hz\n", (T>0? total/T : 0.0));

    free(buf);
    ACTS2200_AI_StopTask(hDev);
    ACTS2200_AI_ReleaseTask(hDev);
    ACTS2200_DEV_Release(hDev);
    fprintf(stderr, "Done.\n");
    return 0;
}